<?php
/**
 * Helper Functions for SMS Processing and Data Provisioning
 */

require_once 'config.php';
require_once 'db_connect.php';

/**
 * Parse payment SMS to extract amount, sender, and transaction ID
 */
function parseSmsMessage($message, $sender) {
    global $PAYMENT_SENDERS;
    
    $parsed = [
        'amount' => null,
        'sender_number' => null,
        'transaction_id' => null,
        'operator' => null,
        'is_payment' => false,
    ];
    
    // Determine operator
    foreach ($PAYMENT_SENDERS as $key => $name) {
        if (stripos($sender, $name) !== false) {
            $parsed['operator'] = $key;
            break;
        }
    }
    
    // Pattern 1: "You have received $X.XX from 615123456. Ref: TXN12345"
    if (preg_match('/received\s+\$?([0-9]+\.?[0-9]*)\s+from\s+([0-9]+)/i', $message, $matches)) {
        $parsed['amount'] = $matches[1];
        $parsed['sender_number'] = $matches[2];
        $parsed['is_payment'] = true;
    }
    
    // Pattern 2: "Payment of $X.XX from 615123456"
    if (!$parsed['is_payment'] && preg_match('/payment\s+of\s+\$?([0-9]+\.?[0-9]*)\s+from\s+([0-9]+)/i', $message, $matches)) {
        $parsed['amount'] = $matches[1];
        $parsed['sender_number'] = $matches[2];
        $parsed['is_payment'] = true;
    }
    
    // Pattern 3: "Transfer confirmed: $X.XX sender: 615123456"
    if (!$parsed['is_payment'] && preg_match('/transfer.*\$?([0-9]+\.?[0-9]*).*(?:sender|from)[:\s]+([0-9]+)/i', $message, $matches)) {
        $parsed['amount'] = $matches[1];
        $parsed['sender_number'] = $matches[2];
        $parsed['is_payment'] = true;
    }
    
    // Extract transaction/reference ID
    if (preg_match('/(?:ref|reference|txn|transaction)[:\s]+([A-Z0-9]+)/i', $message, $matches)) {
        $parsed['transaction_id'] = $matches[1];
    }
    
    // If no specific transaction ID found, use last alphanumeric sequence
    if (!$parsed['transaction_id'] && preg_match('/([A-Z0-9]{6,})/i', $message, $matches)) {
        $parsed['transaction_id'] = $matches[1];
    }
    
    // Normalize amount (remove $ sign, ensure decimal format)
    if ($parsed['amount']) {
        $parsed['amount'] = str_replace('$', '', $parsed['amount']);
        $parsed['amount'] = number_format((float)$parsed['amount'], 2, '.', '');
    }
    
    return $parsed;
}

/**
 * Check if transaction already exists (prevent duplicates)
 */
function transactionExists($conn, $transaction_id) {
    if (empty($transaction_id)) {
        return false;
    }
    
    $sql = "SELECT id FROM received_sms WHERE transaction_id = ? LIMIT 1";
    $result = fetchOne($conn, $sql, [$transaction_id], 's');
    
    return $result !== null;
}

/**
 * Save received SMS to database
 */
function saveSms($conn, $sender, $message, $timestamp, $parsed_data) {
    $sql = "INSERT INTO received_sms 
            (sender, message, timestamp, amount, sender_number, transaction_id, operator, processed, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 0, NOW())";
    
    $params = [
        $sender,
        $message,
        $timestamp,
        $parsed_data['amount'],
        $parsed_data['sender_number'],
        $parsed_data['transaction_id'],
        $parsed_data['operator']
    ];
    
    $stmt = executeQuery($conn, $sql, $params, 'sssssss');
    
    if ($stmt) {
        $insert_id = $conn->insert_id;
        $stmt->close();
        return $insert_id;
    }
    
    return false;
}

/**
 * Get data bundle size based on payment amount
 */
function getDataBundle($amount) {
    global $BUNDLE_MAPPING;
    
    // Normalize amount
    $amount = number_format((float)$amount, 2, '.', '');
    
    if (isset($BUNDLE_MAPPING[$amount])) {
        return $BUNDLE_MAPPING[$amount];
    }
    
    // Also check without decimal
    $amount_int = (string)(int)$amount;
    if (isset($BUNDLE_MAPPING[$amount_int])) {
        return $BUNDLE_MAPPING[$amount_int];
    }
    
    return null;
}

/**
 * Provision data bundle via Operator API
 */
function provisionViaApi($operator, $phone_number, $bundle_size) {
    // This is a placeholder - implement based on your operator's API
    // Each operator has different API specifications
    
    $api_url = '';
    
    switch (strtolower($operator)) {
        case 'hormuud':
            $api_url = HORMUUD_API_URL;
            break;
        case 'somnet':
            $api_url = SOMNET_API_URL;
            break;
        case 'somtel':
            $api_url = SOMTEL_API_URL;
            break;
        default:
            return false;
    }
    
    if (empty($api_url)) {
        return false;
    }
    
    // Prepare API request
    $data = [
        'phone' => $phone_number,
        'bundle' => $bundle_size,
        'timestamp' => date('Y-m-d H:i:s'),
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, API_TIMEOUT);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json',
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        $result = json_decode($response, true);
        return isset($result['success']) && $result['success'];
    }
    
    return false;
}

/**
 * Provision data bundle via SMS Gateway (fallback)
 */
function provisionViaSms($operator, $phone_number, $bundle_size) {
    global $OPERATOR_SHORTCODES;
    
    if (!isset($OPERATOR_SHORTCODES[$operator])) {
        return false;
    }
    
    // Build shortcode SMS (example format, adjust based on actual operator codes)
    $shortcode = $OPERATOR_SHORTCODES[$operator];
    $sms_message = $shortcode . $bundle_size . '#';
    
    // Send via SMS Gateway
    $data = [
        'username' => SMS_GATEWAY_USERNAME,
        'password' => SMS_GATEWAY_PASSWORD,
        'to' => $phone_number,
        'message' => $sms_message,
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, SMS_GATEWAY_URL);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, API_TIMEOUT);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return $http_code == 200;
}

/**
 * Main provisioning function
 */
function provisionDataBundle($conn, $operator, $phone_number, $amount, $sms_id) {
    // Get bundle size
    $bundle_size = getDataBundle($amount);
    
    if (!$bundle_size) {
        logMessage("No bundle mapping found for amount: $amount");
        return [
            'success' => false,
            'message' => 'No bundle mapping for this amount',
        ];
    }
    
    logMessage("Provisioning $bundle_size to $phone_number (operator: $operator)");
    
    $success = false;
    $method = '';
    
    // Try API first if enabled
    if (USE_OPERATOR_API) {
        $success = provisionViaApi($operator, $phone_number, $bundle_size);
        $method = 'API';
    }
    
    // Fallback to SMS gateway
    if (!$success) {
        $success = provisionViaSms($operator, $phone_number, $bundle_size);
        $method = 'SMS';
    }
    
    // Log provisioning attempt
    $sql = "INSERT INTO provision_logs 
            (sms_id, phone_number, operator, bundle_size, amount, method, status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
    
    $status = $success ? 'success' : 'failed';
    
    $params = [
        $sms_id,
        $phone_number,
        $operator,
        $bundle_size,
        $amount,
        $method,
        $status
    ];
    
    executeQuery($conn, $sql, $params, 'issssss');
    
    // Mark SMS as processed
    if ($success) {
        $update_sql = "UPDATE received_sms SET processed = 1 WHERE id = ?";
        executeQuery($conn, $update_sql, [$sms_id], 'i');
    }
    
    return [
        'success' => $success,
        'bundle_size' => $bundle_size,
        'method' => $method,
        'message' => $success ? "Successfully provisioned $bundle_size" : "Failed to provision",
    ];
}

